%Matt Bristow
%09/11/2017

function [Hoff,Hoff_err,Hmid,Hmid_err,Hon,Hon_err] = Find_AllCritFields(X,Y)
%SUMMARY:
%   Function to find the Hoffset, Honset and Hmidpoint values from inputted
%   H (X) and rho (Y) data. This is done by fitting through the midpoint.
%   Intercepts with this fit and fits above and below the axis give Hoff
%   and Hon respectively. Errors are also calculated from fits themselves.
%   Hmid is return as the midpoint of the transition, with a 10% error
%   window.
%   The code should filter out errors by itself (and
%   thus not return silly values for non-real transition),but this has not
%   been extensively tested and may require future adjustments.
%INPUTS:
%   X = single column vector of X data points. This would usually be
%   magnetic field (H).
%   Y = single column vector of Y data points. Must be the same size as X.
%   This would usually be resistivity (rho, in units such as micro Ohm cm).
%OUTPUTS:
%   Hoff = the H_offset field calculated from the intercept in the fits.
%   Hoff_err = the error in the Hoff value calculated.
%   Hmid = the H_midpoint field calculated from the intercept in the fits.
%   Hmid_err = the error in the Hmid value calculated.
%   Hon = the H_onset field calculated from the intercept in the fits.
%   Hon_err = the error in the Hon value calculated.

    szX = size(X);
    szY = size(Y);
    Max_szX = max(szX);
    
    %Check to see if X and Y are the same length
    if(szX(1)~=szY(1))
        disp('Error: dimensions of X and Y do not match!');
        Hoff = NaN(1);
        Hoff_err = NaN(1);
        Hmid = NaN(1);
        Hmid_err = NaN(1);
        Hon = NaN(1);
        Hon_err = NaN(1);
        return;
    end
    
    [X_Max, X_Max_idx] = max(X);
    
    %Find derivative of the data
    step_size = 5;
    dY = Derivative_2(X, Y, step_size);
    
    %Find the maximum in the derivative. This should (approximately)
    %correspond to the midpoint of the transition. We can then check the
    %Hoff value found against this later to clarify if it is correct. Hoff
    %should always be less than Hmid.
    [dY_Max, idx] = max(dY(:,2));
    Hmid = X(idx);
    I = ceil(idx/2);
    
    %Finds the error in Hmid to 10% accuracy.
    i = idx;
    while(dY(i,2)>dY_Max*0.9 && i>2)
        i = i - 1;
    end
    err1 = abs(dY(i,1)-dY(idx,1));
    i = idx;
    while(dY(i,2)>dY_Max*0.9 && i<Max_szX)
        i = i + 1;
    end
    err2 = abs(dY(idx,1)-dY(i,1));
    [Hmid_err, ~] = max([err1,err2]);
    
    %Find a linear fit to the transition region, using points between the
    %full width at half maximum either side of the peak in the derivative.
    i = idx;
    while(dY(i,2)>dY_Max*0.5 && i>2)
        i = i - 1;
    end
    Imid1 = i;
    i = idx;
    while(dY(i,2)>dY_Max*0.5 && i<Max_szX)
        i = i + 1;
    end
    Imid2 = i;
    I1 = Imid2 + 2*(Imid2 - idx);
    I2 = Imid2 + 3*(Imid2 - idx);
    %checks to see if there is any point in continuing. If not, sets values
    %to nan and aborts.
    if(abs(Imid2-Imid1)<5 || Hmid>0.95*X_Max || Imid2>(Max_szX-1) ... 
            || Imid1<5)
        Hoff = NaN(1);
        Hoff_err = NaN(1);
        Hmid = NaN(1);
        Hmid_err = NaN(1);
        Hon = NaN(1);
        Hon_err = NaN(1);
        return;
    end
    
    %Fit around the midpoind with a linear fit.
    [P,S] = polyfit(X(Imid1:Imid2),Y(Imid1:Imid2),1);
    [~, delta] = polyval(P,[X(Imid1:Imid2),Y(Imid1:Imid2)],S);
    fit2 = (P(1)*X) + P(2);
    m2 = P(1);
    [m2_err, ~] = max(delta(:,1));
    c2 = P(2);
    [c2_err, ~] = max(delta(:,2));
    
    %checks to see if there is any point in continuing. If not, sets values
    %to nan and aborts.
    if(m2_err>0.5*abs(m2) || c2_err>0.5*abs(c2))
        Hoff = NaN(1);
        Hoff_err= NaN(1);
        Hmid = NaN(1);
        Hmid_err = NaN(1);
        Hon = NaN(1);
        Hon_err = NaN(1);
        return;
    end
    
    %Iterate through to check for non-values in the low field limit.
    i = 1;
    while(isnan(X(i)) || isnan(Y(i)) && i<Max_szX)
        i = i + 1;
    end
    
    %Fit to the low field region with a linear fit.
    [P,S] = polyfit(X(i:I),Y(i:I),1);
    [~, delta] = polyval(P,[X(i:I),Y(i:I)],S);
    fit1 = (P(1)*X) + P(2);
    m1 = P(1);
    [m1_err, ~] = max(delta(:,1));
    c1 = P(2);
    [c1_err, ~] = max(delta(:,2));
    
    [X_Min, X_Min_idx] = min(X);
    Y_Min = Y(X_Min_idx);
    Y_Max = Y(X_Max_idx);
    
    grad = (Y_Max - Y_Min)/(X_Max - X_Min);
    
    %checks to see if there is any point in continuing. If not, sets values
    %to nan and aborts.
    %if(abs(m1)>0.1*abs(grad) || abs(c1-Y_Min)>2*c1_err)
    if(abs(m1)>0.3*abs(grad) || abs(c1-Y_Min)>10*c1_err)
        Hoff = NaN(1);
        Hoff_err= NaN(1);
        Hmid = NaN(1);
        Hmid_err = NaN(1);
        Hon = NaN(1);
        Hon_err = NaN(1);
        return;
    end
    
    %Find Hoff from the intercept in the fits
    Hoff = (c2 - c1)/(m1 - m2);
    y = c1 + (m1*Hoff);
    
    %Checks to see if it is sensible.
    if(Hoff>Hmid || Hoff<0 || Hoff>X_Max)
        Hoff = NaN(1);
        Hoff_err = NaN(1);
        Hmid = NaN(1);
        Hmid_err = NaN(1);
        Hon = NaN(1);
        Hon_err = NaN(1);
        return;
    end
    
    Hoff_err = Hoff*((((c1_err+c2_err)/(c2-c1))^2) ...
        + (((m1_err+m2_err)/(m1-m2))^2))^0.5;
    
        %Fit to the field region above the transition with a linear fit.
    [P,S] = polyfit(X(I1:I2),Y(I1:I2),1);
    [~, delta] = polyval(P,[X(I1:I2),Y(I1:I2)],S);
    fit3 = (P(1)*X) + P(2);
    m3 = P(1);
    [m3_err, ~] = max(delta(:,1));
    c3 = P(2);
    [c3_err, ~] = max(delta(:,2));
    
    %Find Hon from the intercept in the fits
    Hon = (c2 - c3)/(m3 - m2);
    y2 = c3 + (m3*Hon);
    
    %Checks to see if it is sensible.
    if(Hon<Hmid || Hon<=0 || Hon>X_Max)
        Hon = NaN(1);
        Hon_err = NaN(1);
        Hmid = NaN(1);
        Hmid_err = NaN(1);
        return;
    end
    
    Hon_err = Hon*((((c3_err+c2_err)/(c2-c3))^2) ...
        + (((m3_err+m2_err)/(m3-m2))^2))^0.5;
    
    Ilow = Imid1 - 2*(idx - Imid1);
    
    figure; hold on;
    plot(X,Y,'-','color','k','linewidth',1);
    plot(X(Ilow:I1),fit2(Ilow:I1),'--','color','r','linewidth',1);
    plot(X(1:Imid2),fit1(1:Imid2),'--','color','g','linewidth',1);
    plot(X(Imid1:I2),fit3(Imid1:I2),'--','color','g','linewidth',1);
    plot(Hoff,y,'*','color','k','markersize',3);
    plot(Hon,y2,'*','color','k','markersize',3);
    set(gca,'fontsize',15,'fontname','times');
    set(gcf,'color','w');
    xlabel('\it x \rm','fontsize',20,'fontname','times');
    ylabel('\it y \rm','fontsize',20,'fontname','times');
    %axis([X_Min,X_Max,10*floor(Y_Min/10.0),10*ceil(Y_Max/10.0)]);
    box on;
%     Legend = cell(3,1);
%     Legend{1} = strcat('Input Data');
%     Legend{2} = strcat('Fit Region');
%     Legend{3} = strcat('Fit');
%     legend(Legend, 'location', 'NorthWest');
%     set(legend,'FontSize',15);
%     legend show Location NorthWest
%     legend boxoff
    hold off;
end

